package ga.core.selection;

import ga.core.individual.IIndividual;
import ga.core.individual.IndividualList;
import ga.core.individual.population.IPopulation;

/**
 * Implementation of a tournament selector. This will take a {@code n} random
 * individuals and only the best of these {@code n} individuals will be
 * selected.
 * <p />
 * {@code n} is the tournament size.
 * 
 * @param <T>
 *          The generic type of individuals.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public class TournamentSelector<T extends IIndividual<T>> implements
    ISelector<T> {
  private static final int SELECTED_INDIVIDUALS = 2;

  private int tournamentSize = 3;
  private boolean replaceOldOnInsert = true;
  private final boolean allowDuplicates = true;

  /**
   * Create a new tournament selector with the given tournament size.
   * 
   * @param tournamentSize
   *          The tournament size.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public TournamentSelector(final int tournamentSize) {
    this.tournamentSize = tournamentSize;
  }

  @Override
  public IndividualList<T> select(final IPopulation<T> population) {
    // a selection is not possible
    if (population.size() < tournamentSize) {
      throw new RuntimeException("There are not enough individuals to select");
    }

    final IndividualList<T> list = new IndividualList<T>();

    while (list.size() <= tournamentSize) {
      final T ind = population.getRandomIndividualForSelection();

      if ((allowDuplicates || !list.contains(ind)) && ind.isEvaluated()) {
        list.add(ind);
      }
    }

    // sort by fitness
    list.sort(false);

    // remove the first individuals until list size is 2
    while (list.size() > SELECTED_INDIVIDUALS) {
      list.remove(0);
    }

    if (list.size() != SELECTED_INDIVIDUALS) {
      throw new RuntimeException("Wrong list size");
    }

    // be sure to handle only clones!
    for (int i = 0; i < list.size(); i++) {
      list.set(i, list.get(i).clone());
    }

    return list;
  }

  /**
   * Activate/deactivate the behavior of replacing individuals during insertion.
   * If deactivated, the population will grow with each insertion.
   * 
   * @param replaceOldOnInsert
   *          Replace or not.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void setReplaceOldOnInsert(final boolean replaceOldOnInsert) {
    this.replaceOldOnInsert = replaceOldOnInsert;
  }

  /**
   * Sets the tournament size.
   * 
   * @param tournamentSize
   *          The tournament size.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void setTournamentSize(final int tournamentSize) {
    this.tournamentSize = tournamentSize;
  }

  /**
   * Getter for the tournament size.
   * 
   * @return The tournament size.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public int getTournamentSize() {
    return tournamentSize;
  }

  @Override
  public void insert(final IndividualList<T> list,
      final IPopulation<T> population, final boolean useEliteStrategy) {

    if (list.size() != SELECTED_INDIVIDUALS) {
      throw new RuntimeException("Wrong list size");
    }

    if (replaceOldOnInsert) {
      // inverse sort by fitness
      population.getIndividuals().sort(false);

      // this already implies elitism
      for (int i = 0; i < list.size(); i++) {
        population.getIndividuals().set(i, list.get(i));
      }
    } else {
      population.addIndividuals(list);
    }
  }

}
